#!/bin/sh


########################################
# description:
#     Check if specified ip is valid.
#
# return:
#     0 : valid
#     1 : invalid
#
# example:
#     is_ip_valid 10.1.1.1
#     echo ${?}
########################################
function is_ip_valid() {
    local ip_address=$1

    # chek: 4 segments
    local ip_segments_count=`echo $ip_address | awk -F '.' '{print NF}'`
    [ "$ip_segments_count" -ne "4" ] && return 1

    local ip_segment_1=`echo $ip_address | awk -F '.' '{print $1}'`
    local ip_segment_2=`echo $ip_address | awk -F '.' '{print $2}'`
    local ip_segment_3=`echo $ip_address | awk -F '.' '{print $3}'`
    local ip_segment_4=`echo $ip_address | awk -F '.' '{print $4}'`

    # check range 0~255 for each segment
    for seg in "$ip_segment_1" "$ip_segment_2" "$ip_segment_3" "$ip_segment_4"
    do
        [ "$seg" -lt 0 -o "$seg" -gt 255 ] 1> /dev/null 2>&1 && return 1
        [ "${?}" -eq 2 ] && return 1
    done

    # check not start with 0 for each segment and no empty space
    for seg in "$ip_segment_1" "$ip_segment_2" "$ip_segment_3" "$ip_segment_4"
    do
        echo "$seg" | grep "^0[0-9]\{1,2\}" 1>/dev/null 2>&1
        [ "${?}" -eq 0 ] && return 1

        echo "$seg" | grep " " 1>/dev/null 2>&1
        [ "${?}" -eq 0 ] && return 1
    done
    
    # check segment 1: should be greater than 0
    [ "$ip_segment_1" -lt 1 ] 1> /dev/null 2>&1 && return 1
    [ "${?}" -eq 2 ] && return 1

    return 0
}


########################################
# description:
#     Check if specified username is valid
#
# return:
#     0 : valid
#     1 : invalid
#
# example:
#     is_username_valid John
#     echo ${?}
########################################
function is_username_valid() {

    # check if username contains "space" character
    local column_count=`echo $1 | awk '{print NF}'`
    [ "$column_count" -ne "1" ] && return 1
    
    return 0
}


########################################
# description:
#     Check if specified password is valid
#
# return:
#     0 : valid
#     1 : invalid
#
# example:
#     is_password_valid xxxxxx
#     echo ${?}
########################################
function is_password_valid() {

	# allow all passwords
	return 0
}


########################################
# description
#     Check if specified integer is positive and valid
#
# return:
#     0 : positive and valid
#     1 : negtive or invalid
#
# example:
#     is_integer_positive_and_valid 12345
#     echo ${?}
########################################
function is_integer_positive_and_valid() {

    # check if the integer > 0 
    [ "$1" -le 0 ] 1> /dev/null 2>&1 && return 1
    
    # when it is not a valid integer, bash will return 2 as error code
    [ "${?}" -eq 2 ] && return 1
    
    return 0
}


########################################
# description
#     Check if specified port number is valid
#
# return:
#     0 : valid
#     1 : invalid
#
# example:
#     is_port_number_valid 12345
#     echo ${?}
########################################
function is_port_number_valid() {
	is_integer_positive_and_valid "$1"
	[ "${?}" -eq 1 ] && return 1
	
	[ "$1" -gt 65535 ] && return 1
	
	return 0
}

########################################
# description
#     Check if specified netmask is valid
#
# return:
#     0 : valid
#     1 : invalid
#
# example:
#     is_netmask_valid 255.255.255.0
#     echo ${?}
########################################
function is_netmask_valid() {
    local netmask=$1

    if [ "$netmask" == "0.0.0.0" ];then
        #dprint 0 "info" "Invalid netmask: $1"
        return 1
    fi
    
    if [ "$netmask" == "255.255.255.255" ];then
        #dprint 0 "info" "Invalid netmask: $1"
        return 1
    fi

    # when netmask is an invalid ip
    is_ip_valid "$netmask"
    [ "${?}" -eq 1 ] && return 1
    

    # get binary format of the netmask
    local binary_number=""
    local ip_segments=`echo $netmask | sed 's/\./ /g'`
    for segment in $ip_segments
    do
        local tmp=""
        for (( i=1; i<=8; i=i+1 ))
        do
            tmp="$(( $segment % 2 ))$tmp"
            segment="$(( $segment / 2 ))"
        done

        binary_number="$binary_number$tmp"
    done

    
    # when pattern "01" is found, it is not a valid netmask
    local result=`echo $binary_number | grep "01"`
    [ -n "$result" ] && return 1
    
    return 0
}


########################################
# description
#     Check if specified hostname is valid
#
# return:
#     0 : valid
#     1 : invalid
#
# example:
#     is_hostname_valid abcdefg
#     echo ${?}
########################################
function is_hostname_valid() {
    local hostname=$1

    
    # only 1 ~ 64 characters
	local length=`echo "$hostname" | wc -c`
	[ "$length" -lt 2 -o "$length" -gt 65 ] && return 1
    
    if [ "$(echo "$hostname" | cut -b 1-1)" == "-" ]; then
        return 1
    fi
    
    # no space
    local tmp=`echo "$hostname" | grep " "`
    [ -n "$tmp" ] && return 1
    
    
    # only a-z, A-Z, 0-9, ., and - 
    tmp=`echo "$hostname" | grep "^[a-zA-Z0-9\.-]\{1,\}$"`
    [ -z "$tmp" ] && return 1
    
    
    return 0
}

########################################
# description:
#     Check if another instance is running.
# [Warning] 
#     This function checks if another instance 
#   running by checking if the "basename" of 
#   execution script is in running process list.
#   This approach is much simpler and does not 
#   need to maintain a lock file and its expiry
#   timestamp.
#     Before using this function, you have to 
#   make sure no other script have the same base 
#   name as the calling script will possible be 
#   running. 
#     If the name of the calling script is too
#   normal, such as run.sh, setup.sh..., you 
#   have to be careful!
#
# return:
#     0 : No, no other instance is running.
#     1 : Yes, another instance is running.
#
# example:
#     is_another_instance_running
#     echo ${?}
########################################
is_another_instance_running(){
	me=`basename $0`
	if [ $(pidof -x ${me} | wc -w) -gt 2 ]; then 
		return 1
	fi

	return 0
}



########################################
# description:
#     Check if specified ip is conflict with broadcast IP
#
# return:
#     0 : no conflict
#     1 : conflict
#
# example:
#     is_conflict_broadcast_ip 10.1.1.1
#     echo ${?}
########################################
function is_conflict_broadcast_ip() {
    local input=$1
    if [ "$input" == "255.255.255.255" ];then
        #dprint 0 "info" "IP conflict with broadcast IP: $input"
        return 1
    fi

    return 0
}

########################################
# description:
#     Check if specified ip is conflict with multicast IP
#
# return:
#     0 : no conflict
#     1 : conflict
#
# example:
#     is_conflict_multicast_ip 10.1.1.1
#     echo ${?}
########################################
function is_conflict_multicast_ip() {
    local input=$1
    is_ip_in_range $input "224.0.0.0" "239.255.255.255"
    if [ "${?}" -eq 1 ];then
        #dprint 0 "info" "IP conflict with multicast IP: $input"
        return 1
    fi

    return 0
}

########################################
# description:
#     Check if specified ip is conflict with link-local IP
#
# return:
#     0 : no conflict
#     1 : conflict
#
# example:
#     is_conflict_linklocal_ip 10.1.1.1
#     echo ${?}
########################################
function is_conflict_linklocal_ip() {
    local input=$1
    is_ip_in_range $input "169.254.0.0" "169.254.255.255"
    if [ "${?}" -eq 1 ];then
         #dprint 0 "info" "IP conflict with link-local IP: $input"
         return 1
    fi
    return 0
}

########################################
# description:
#     Check if specified ip is conflict with 240.0.0.0 V 255.255.255.254
#
# return:
#     0 : no conflict
#     1 : conflict
#
# example:
#     is_conflict_240_ip_rfc5735 10.1.1.1
#     echo ${?}
########################################
function is_conflict_240_ip_rfc5735() {
    local input=$1
    is_ip_in_range $input "240.0.0.0" "255.255.255.254"
    if [ "${?}" -eq 1 ];then
        #dprint 0 "info" "IP conflict with 240.0.0.0 V 255.255.255.254 IP: $input"
        return 1
    fi

    return 0
}

########################################
# description:
#     Check if specified ip is conflict with 127.0.0.0 V 127.255.255.255
#
# return:
#     0 : no conflict
#     1 : conflict
#
# example:
#     is_conflict_loopback_ip_rfc5735 10.1.1.1
#     echo ${?}
########################################
function is_conflict_loopback_ip_rfc5735() {
    local input=$1
    is_ip_in_range $input "127.0.0.0" "127.255.255.255"
    if [ "${?}" -eq 1 ];then
        #dprint 0 "info" "IP conflict with loopback IP: $input"
        return 1
    fi

    return 0
}

########################################
# description:
#     Check if specified ip is valid for management IP
#
# return:
#     0 : valid
#     1 : invalid
#
# example:
#     is_ip_address_valid 10.1.1.1
#     echo ${?}
########################################
function is_ip_address_valid() {
    local input=$1

    is_ip_valid "$input"
    [ "${?}" -eq 1 ] && return 1

    is_conflict_broadcast_ip "$input"
    [ "${?}" -eq 1 ] && return 1

    is_conflict_multicast_ip "$input"
    [ "${?}" -eq 1 ] && return 1

    is_conflict_linklocal_ip "$input"
    [ "${?}" -eq 1 ] && return 1

    is_conflict_240_ip_rfc5735 "$input" #240.0.0.0 V 255.255.255.254
    [ "${?}" -eq 1 ] && return 1

    is_conflict_loopback_ip_rfc5735 "$input" #127.0.0.0 V 127.255.255.255
    [ "${?}" -eq 1 ] && return 1

    return 0
}

function ip_to_binary_str() {
    local input=$1
    # get binary format of the IP
    local binary_number=""
    local ip_segments=`echo $input | sed 's/\./ /g'`
    for segment in $ip_segments
    do
        local tmp=""
        for (( i=1; i<=8; i=i+1 ))
        do
            tmp="$(( $segment % 2 ))$tmp"
            segment="$(( $segment / 2 ))"
        done

        binary_number="$binary_number$tmp"
    done
    echo $binary_number
}

########################################
# description:
#     Check if the two ip are in the same subnet by netmask
#
# return:
#     0 : same 
#     1 : not same 
#
# example:
#     are_in_the_same_subnet ip1 netmask1 ip2 netmask2
#     echo ${?}
########################################
function are_in_the_same_subnet() {
    local ip1=$1
    local netmask1=$2
    local ip2=$3
    local netmask2=$4

    #dprint 0 "debug" "IP1: ${ip1}"
    #dprint 0 "debug" "Netmask1: ${netmask1}"
    #dprint 0 "debug" "IP2: ${ip2}"
    #dprint 0 "debug" "Netmask2: ${netmask2}"

    is_ip_address_valid ${ip1}
    if [ "${?}" -eq 1 ]; then
        #dprint 0 "info" "invalid ip1 checked by is_ip_address_valid: ${ip1}"
        return 1
    fi 

    is_netmask_valid ${netmask1}
    if [ "${?}" -eq 1 ]; then
        #dprint 0 "info" "invalid netmask1 checked by is_netmask_valid: ${netmask1}"
        return 1
    fi 

    is_ip_address_valid ${ip2}
    if [ "${?}" -eq 1 ]; then
        #dprint 0 "info" "invalid ip2 checked by is_ip_address_valid: ${ip2}"
        return 1
    fi 

    is_netmask_valid ${netmask2}
    if [ "${?}" -eq 1 ]; then
        #dprint 0 "info" "invalid netmask2 checked by is_netmask_valid: ${netmask2}"
        return 1
    fi 

    #check if ip1 and ip2 are in the same subnet by netmask
    local bin_ip1=`ip_to_binary_str "${ip1}"`
    #dprint 0 "debug" "IP1: ${ip1} -> ${bin_ip1}"

    local bin_netmask1=`ip_to_binary_str "${netmask1}"`
    #dprint 0 "debug" "Netmask1: ${netmask1} -> ${bin_netmask1}"

    local bin_ip2=`ip_to_binary_str "${ip2}"`
    #dprint 0 "debug" "IP2: ${ip2} -> ${bin_ip2}"

    local bin_netmask2=`ip_to_binary_str "${netmask2}"`
    #dprint 0 "debug" "Netmask2: ${netmask2} -> ${bin_netmask2}"

    local NUM1=`echo ${bin_netmask1} | sed -e "s/0//g" | wc -m`
    NUM1=`expr ${NUM1} - 1` # count how many 1 in netmask
    #dprint 0 "debug" "netmask1 number: ${NUM1}"

    local NUM2=`echo ${bin_netmask2} | sed -e "s/0//g" | wc -m`
    NUM2=`expr ${NUM2} - 1` # count how many 1 in netmask
    #dprint 0 "debug" "netmask2 number: ${NUM2}"

    local bin_ip1_sub=${bin_ip1:0:$NUM1}
    local bin_ip2_sub=${bin_ip2:0:$NUM2}

    if [ "${bin_ip1_sub}" == "${bin_ip2_sub}" ];then
        #dprint 0 "debug" "${ip1}/${NUM1} and ${ip2}/${NUM2} are in the same subnet"
        return 0
    fi

    #dprint 0 "info" "${ip1}/${NUM1} and ${ip2}/${NUM2} are not in the same subnet"
    return 1
}



########################################
# description:
#     Check if specified gateway is valid
#
# return:
#     0 : valid
#     1 : invalid
#
# example:
#     is_gateway_valid ip netmask gateway
#     echo ${?}
########################################
function is_gateway_valid() {
    local ip=$1
    local netmask=$2
    local gateway=$3

    #dprint 0 "debug" "IP: $ip"
    #dprint 0 "debug" "Netmask: $netmask"
    #dprint 0 "debug" "Gateway: $gateway"

    if [ "$ip" == "$gateway" ]; then
        #dprint 0 "info" "invalid gateway IP=Gateway: $gateway"
        return 1
    fi

    is_ip_address_valid $gateway
    if [ "${?}" -eq 1 ]; then
        #dprint 0 "info" "invalid gateway checked by is_ip_address_valid: $gateway"
        return 1
    fi 

    #check if ip and gateway are in the same subnet by netmask
    local bin_ip=`ip_to_binary_str "$ip"`
    #dprint 0 "debug" "IP: $ip -> $bin_ip"

    local bin_netmask=`ip_to_binary_str "$netmask"`
    #dprint 0 "debug" "Netmask: $netmask-> $bin_netmask"

    local bin_gateway=`ip_to_binary_str "$gateway"`
    #dprint 0 "debug" "Gateway: $gateway -> $bin_gateway"

    local NUM=`echo $bin_netmask | sed -e "s/0//g" | wc -m`
    NUM=`expr $NUM - 1` # count how many 1 in netmask
    #dprint 0 "debug" "netmask number: $NUM"

    local bin_ip_sub=${bin_ip:0:$NUM}
    local bin_gateway_sub=${bin_gateway:0:$NUM}

    if [ "$bin_ip_sub" == "$bin_gateway_sub" ];then
        #dprint 0 "debug" "valid gateway: $1"
        return 0
    fi

    #dprint 0 "info" "invalid gateway: $bin_ip_sub $bin_gateway_sub"
    return 1
}

inet_aton ()
{
    local IFS=. ipaddr ip32 i
    local ipaddr=($1)
    for i in 3 2 1 0
    do
        (( ip32 += ipaddr[3-i] * (256 ** i) ))
    done

    echo $ip32
}

########################################
# description
#     Check if specified IP is in given range
#
# return:
#     0 : no, not in range
#     1 : yes, in range
#
# example:
#     is_ip_in_range ip min max
#     echo ${?}
########################################
function is_ip_in_range() {
    local ip=`inet_aton $1`
    local min=`inet_aton $2`
    local max=`inet_aton $3`

    if [[ "$ip" -lt "$min" || "$ip" -gt "$max" ]]; then
       return 0
    fi

    return 1
}

########################################
# description
#     Check if specified fqdn is valid
#
# return:
#     0 : valid
#     1 : invalid
#
# example:
#     is_fqdn_valid abcdefg
#     echo ${?}
########################################
function is_fqdn_valid() {
    local fqdn=$1

    # total length at least 1 and up to 255
	local length=`echo "$fqdn" | wc -c`
	[ "$length" -lt 2 -o "$length" -gt 256 ] && return 1

    # replace "." with " " which is default delimiter
    local fqdn_labels=`echo $fqdn | sed 's/\./ /g'`
    
    for label in $fqdn_labels
    do
        # each label at least 1 and up to 63 char
        length=`echo "$label" | wc -c`
        [ "$length" -lt 2 -o "$length" -gt 64 ] && return 1

        # only a-z, A-Z, 0-9, and - 
        local tmp=`echo "$label" | grep "^[a-zA-Z0-9-]\{1,\}$"`
        [ -z "$tmp" ] && return 1
    done

    return 0
}

########################################
# description:
#     Check if specified ipv6 is valid for management IP
#
# return:
#     0 : valid
#     1 : invalid
#
# example:
#     is_ipv6_address_valid 2620:101:4002:771::1
#     echo ${?}
########################################
function is_ipv6_address_valid() {
    local input=$1
    local RET=0

    ipcalc -cs -6 $input
    if [ "${?}" -ne 0 ]; then
        RET=1
    else
        local output=$(ipv6calc --in ipv6addr $input --out ipv6addr)
        if [ "$output" = "" ] || [ "$output" = "::" ] || [ "$output" = "::1" ]; then
            RET=1
        fi
    fi

    return $RET
}

########################################
# description:
#     Check if specified ipv6 subnet prefix length is valid for management IP
#
# return:
#     0 : valid
#     1 : invalid
#
# example:
#     is_ipv6_prefix_length_valid 64 
#     echo ${?}
########################################
function is_ipv6_prefix_length_valid() {
    local input=$1

    if [ "$input" = "" ] || [ ! -n "$(echo $input | sed -n "/^[0-9]\+$/p")" ] || [ $input -lt 1 ] || [ $input -gt 127 ]; then
        return 1
    else
        return 0
    fi
}

########################################
# description:
#     Check if two given ipv6 addresses are in the same subnet
#
# parameters:
#    ip1 ip2 prefix
#
# return:
#     0 : same
#     1 : not same
########################################
function are_in_the_same_ipv6_subnet() {
    local ip1=$1
    local ip2=$2
    local prefix=$3

    local subnet1=`/usr/bin/sipcalc ${ip1}/${prefix} | grep "^Subnet prefix" | awk -F'-' '{print $2}' | awk '{print $1}'`
    local subnet2=`/usr/bin/sipcalc ${ip2}/${prefix} | grep "^Subnet prefix" | awk -F'-' '{print $2}' | awk '{print $1}'`
    if [ "${subnet1}" = "" ] || [ "${subnet2}" = "" ] || [ "${subnet1}" != "${subnet2}" ]; then
        return 1
    else
        return 0
    fi
}


module_name="DDA"
facility_id="local0"
priority_tag="[   ERROR]"
function convert_to_syslog_info() {
    local id="${1}"
    local priority="${2}"
            
    case $id in 
           "0" ):
                module_name="DDA"
                facility_id="local0"
                ;;
           "1" ):
                module_name="ScanFlow"
                facility_id="local1"
                ;;
           "2" ):
                module_name="service4tmcm"
                facility_id="local2"
                ;;
           "3" ):
                module_name="Cluster"
                facility_id="local3"
                ;;
           "4" ):
                module_name="service4product"
                facility_id="local4"
                ;;
    esac

    declare -l priority="${priority}"
    case $priority in
        "debug" ):
            priority_tag="[    DEBUG]"
            ;;

        "info" ):
            priority_tag="[     INFO]"
            ;;

        "notice"):
            priority_tag="[   NOTICE]"
             ;;

        "warning"):
            priority_tag="[  WARNING]"
             ;;

        "err"):
            priority_tag="[    ERROR]"
             ;;

        "crit"):
            priority_tag="[ CRITICAL]"
             ;;

        "alert"):
            priority_tag="[    ALERT]"
            ;;

        "emerg"):
            priority_tag="[    EMERG]"
            ;;

        * )
            priority_tag="[    ERROR]"
            ;;
     esac
 }
